using System;
using System.Collections.Generic;
using System.IO;
using System.Runtime.InteropServices;
using System.Security.Permissions;
using System.Windows.Forms;

namespace DriveMgr
{
    public partial class MainForm : Form
    {
        List<DriveInfo> drives;

        const int WM_DEVICECHANGE = 0x0219;
        const int DBT_DEVICEARRIVAL = 0x8000; // system detected a new device
        const int DBT_DEVICEREMOVECOMPLETE = 0x8004; //device was removed
        const int DBT_DEVNODES_CHANGED = 0x0007; //device changed
        const int DBT_DEVTYP_VOLUME = 0x00000002; // logical volume

        [SecurityPermission(SecurityAction.LinkDemand, Flags = SecurityPermissionFlag.UnmanagedCode)]
        protected override void WndProc(ref Message m)
        {
            if (m.Msg == WM_DEVICECHANGE && m.WParam.ToInt32() == DBT_DEVICEARRIVAL || m.WParam.ToInt32() == DBT_DEVICEREMOVECOMPLETE || m.WParam.ToInt32() == DBT_DEVNODES_CHANGED)
            {
                if (m.WParam.ToInt32() != DBT_DEVNODES_CHANGED)
                {
                    int devType = Marshal.ReadInt32(m.LParam, 4);
                    if (devType == DBT_DEVTYP_VOLUME)
                    {
                        if (autoUpdateMenuItem.Checked)
                            LoadDrives();
                    }
                }
                else
                {
                    if (autoUpdateMenuItem.Checked)
                        LoadDrives();
                }
            }

            base.WndProc(ref m);
        }

        public MainForm()
        {
            InitializeComponent();
            driveListView.ContextMenu = driveContextMenu;
            drives = new List<DriveInfo>();
        }

        void LoadDrives()
        {
            driveListView.Items.Clear();
            drives.Clear();
            foreach (DriveInfo drive in DriveInfo.GetDrives())
            {
                drives.Add(drive);
                ListViewItem lvi = new ListViewItem(drive.Name);
                if (drive.IsReady)
                {
                    lvi.SubItems.Add(drive.VolumeLabel);
                    lvi.SubItems.Add(drive.DriveType.ToString());
                    lvi.SubItems.Add(drive.TotalSize.ToString());
                    lvi.SubItems.Add(Math.Round((float)drive.AvailableFreeSpace / (float)drive.TotalSize * 100f).ToString() + "%");
                    lvi.SubItems.Add(drive.DriveFormat);
                }
                else
                {
                    lvi.SubItems.Add("<Not ready>");
                    try
                    {
                        lvi.SubItems.Add(drive.DriveType.ToString());
                    }
                    catch (Exception) { }
                }

                if (!(!drive.IsReady && !showNonReadyDrivesMenuItem.Checked))
                    driveListView.Items.Add(lvi);
            }
        }

        private void MainForm_Load(object sender, EventArgs e)
        {
            LoadDrives();
        }

        private void refreshMenuItem_Click(object sender, EventArgs e)
        {
            LoadDrives();
        }

        private void driveListView_DoubleClick(object sender, EventArgs e)
        {
            if (driveListView.SelectedItems.Count > 0 && driveListView.SelectedItems[0] != null)
            {
                new PropertyDialog(drives[driveListView.SelectedIndices[0]]).ShowDialog(this);
            }
        }

        private void propertiesMenuItem_Click(object sender, EventArgs e)
        {
            if (driveListView.SelectedItems.Count > 0 && driveListView.SelectedItems[0] != null)
            {
                new PropertyDialog(drives[driveListView.SelectedIndices[0]]).ShowDialog(this);
            }
        }

        private void driveListView_SelectedIndexChanged(object sender, EventArgs e)
        {
            propertiesMenuItem.Enabled = driveListView.SelectedItems.Count > 0 && driveListView.SelectedItems[0] != null;
        }

        private void aboutMenuItem_Click(object sender, EventArgs e)
        {
            new AboutForm().ShowDialog(this);
        }

        private void autoUpdateMenuItem_Click(object sender, EventArgs e)
        {
            autoUpdateMenuItem.Checked = !autoUpdateMenuItem.Checked;
        }

        private void showNonReadyDrivesMenuItem_Click(object sender, EventArgs e)
        {
            showNonReadyDrivesMenuItem.Checked = !showNonReadyDrivesMenuItem.Checked;
            LoadDrives();
        }

        private void exitMenuItem_Click(object sender, EventArgs e)
        {
            Close();
        }
    }
}
